using System;
using ConnectorLib.JSON;
using Random = UnityEngine.Random;

namespace CrowdControl.Delegates.Effects.Implementations
{
    [Effect("skill_random_0", "skill_random_max", "skill_random_5", "skill_random_-5")]
    public class PlayerSkills : Effect
    {
        public PlayerSkills(CrowdControl mod, NetworkClient client) : base(mod, client) { }
        
        private SkillID? GetRandomSkill(ReadOnlySpan<SkillID?> excludeSkills)
        {
            SkillID randomSkill = (SkillID)Random.Range(0, (int)SkillID.NUM_SKILLS);
            for (int i = 0; i < (int)SkillID.NUM_SKILLS; i++)
            {
                randomSkill = (SkillID)(((int)randomSkill + 1) % (int)SkillID.NUM_SKILLS);
                if (!excludeSkills.Contains(randomSkill)) return randomSkill;
            }
            return null;
        }

        public override EffectResponse Start(EffectRequest request)
        {
            const int NUM_SKILLS = (int)SkillID.NUM_SKILLS;
            PlayerController player = Manager.main.player;
            if (player == null) return EffectResponse.Failure(request.ID, StandardErrors.PlayerNotFound);
            Span<SkillID?> excludeSkills = stackalloc SkillID?[NUM_SKILLS];
            for (int i = 0; i < NUM_SKILLS; i++) excludeSkills[i] = null;
            //SkillID randomSkill = (SkillID)Random.Range(0, (int)SkillID.NUM_SKILLS);

            switch (request.code)
            {
                case "skill_random_0":
                {
                    for (int i = 0; i < NUM_SKILLS; i++)
                    {
                        SkillID? randomSkill = GetRandomSkill(excludeSkills);
                        if (randomSkill == null) break;

                        int skillValue = Manager.saves.GetSkillValue(randomSkill.Value);
                        int skillLevel = SkillExtensions.GetLevelFromSkill(randomSkill.Value, skillValue);
                        
                        if (skillLevel <= 0) continue;
                        
                        player.SetSkillLevel(randomSkill.Value, 0);
                        excludeSkills[i] = randomSkill;
                        return EffectResponse.Success(request.ID);
                    }
                    return EffectResponse.Failure(request.ID, StandardErrors.AlreadyMaximum, "All skill levels already at minimum.");
                }
                case "skill_random_max":
                {
                    for (int i = 0; i < NUM_SKILLS; i++)
                    {
                        SkillID? randomSkill = GetRandomSkill(excludeSkills);
                        if (randomSkill == null) break;

                        int skillValue = Manager.saves.GetSkillValue(randomSkill.Value);
                        int skillLevel = SkillExtensions.GetLevelFromSkill(randomSkill.Value, skillValue);
                        int maxSkillLevel = SkillExtensions.GetMaxSkillLevel(randomSkill.Value);
                        
                        if (skillLevel >= maxSkillLevel) continue;
                        
                        player.SetSkillLevel(randomSkill.Value, maxSkillLevel);
                        excludeSkills[i] = randomSkill;
                        return EffectResponse.Success(request.ID);
                    }
                    return EffectResponse.Failure(request.ID, StandardErrors.AlreadyMaximum, "All skill levels already at maximum.");
                }
                case "skill_random_5":
                {
                    for (int i = 0; i < NUM_SKILLS; i++)
                    {
                        SkillID? randomSkill = GetRandomSkill(excludeSkills);
                        if (randomSkill == null) break;

                        int skillValue = Manager.saves.GetSkillValue(randomSkill.Value);
                        int skillLevel = SkillExtensions.GetLevelFromSkill(randomSkill.Value, skillValue) + 5;

                        if (skillLevel > SkillExtensions.GetMaxSkillLevel(randomSkill.Value))
                            continue;

                        player.SetSkillLevel(randomSkill.Value, skillLevel);
                        excludeSkills[i] = randomSkill;
                        return EffectResponse.Success(request.ID);
                    }
                    return EffectResponse.Failure(request.ID, StandardErrors.AlreadyMaximum, "All skill levels too close to maximum.");
                }
                case "skill_random_-5":
                {
                    for (int i = 0; i < NUM_SKILLS; i++)
                    {
                        SkillID? randomSkill = GetRandomSkill(excludeSkills);
                        if (randomSkill == null) break;

                        int skillValue = Manager.saves.GetSkillValue(randomSkill.Value);
                        int skillLevel = SkillExtensions.GetLevelFromSkill(randomSkill.Value, skillValue) - 5;

                        if (skillLevel < 0)
                            continue;

                        player.SetSkillLevel(randomSkill.Value, skillLevel);
                        excludeSkills[i] = randomSkill;
                        return EffectResponse.Success(request.ID);
                    }
                    return EffectResponse.Failure(request.ID, StandardErrors.AlreadyMinimum, "All skill levels too close to minimum.");
                }
                default:
                    return EffectResponse.Failure(request.ID, StandardErrors.EffectUnknown);
            }
        }
    }
}